<?php
session_start();

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../api/config.php';
$conn = getDBConnection();

// Ensure reported_products table exists
try {
    $conn->exec("CREATE TABLE IF NOT EXISTS reported_products (
        id INT AUTO_INCREMENT PRIMARY KEY,
        product_id INT NOT NULL,
        reporter_id INT NULL,
        reason TEXT NOT NULL,
        status ENUM('pending', 'reviewed', 'dismissed') DEFAULT 'pending',
        admin_note TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        reviewed_at TIMESTAMP NULL,
        FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
        FOREIGN KEY (reporter_id) REFERENCES users(id) ON DELETE SET NULL,
        INDEX idx_product (product_id),
        INDEX idx_reporter (reporter_id),
        INDEX idx_status (status)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
} catch (Exception $e) {
    // ignore
}

// Handle status update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $reportId = isset($_POST['report_id']) ? (int)$_POST['report_id'] : 0;
    $status = $_POST['status'] ?? '';
    $adminNote = trim($_POST['admin_note'] ?? '');

    if ($reportId > 0 && in_array($status, ['pending', 'reviewed', 'dismissed'], true)) {
        $stmt = $conn->prepare(
            "UPDATE reported_products 
             SET status = ?, admin_note = ?, reviewed_at = NOW() 
             WHERE id = ?"
        );
        $stmt->execute([$status, $adminNote, $reportId]);
        header('Location: reports.php?updated=1');
        exit;
    }
}

// Filters
$statusFilter = $_GET['status'] ?? '';

$where = [];
$params = [];

if ($statusFilter !== '' && in_array($statusFilter, ['pending', 'reviewed', 'dismissed'], true)) {
    $where[] = "r.status = ?";
    $params[] = $statusFilter;
}

$whereSql = '';
if (!empty($where)) {
    $whereSql = 'WHERE ' . implode(' AND ', $where);
}

$stmt = $conn->prepare(
    "SELECT r.*, 
            p.title AS product_title,
            p.status AS product_status,
            u.name AS reporter_name,
            u.phone AS reporter_phone
     FROM reported_products r
     LEFT JOIN products p ON r.product_id = p.id
     LEFT JOIN users u ON r.reporter_id = u.id
     $whereSql
     ORDER BY r.created_at DESC"
);
$stmt->execute($params);
$reports = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reported Ads - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background: #2c3e50;
        }
        .sidebar a {
            color: #ecf0f1;
            text-decoration: none;
            padding: 15px;
            display: block;
        }
        .sidebar a:hover {
            background: #34495e;
        }
        .sidebar a.active {
            background: #3498db;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar p-0">
                <div class="p-3">
                    <h4 class="text-white">Mathbaria Shop</h4>
                    <p class="text-white-50 small">Admin Panel</p>
                </div>
                <nav>
                    <a href="index.php"><i class="bi bi-house"></i> Dashboard</a>
                    <a href="products.php"><i class="bi bi-box"></i> Products</a>
                    <a href="pending.php"><i class="bi bi-clock-history"></i> Pending Approval</a>
                    <a href="reports.php" class="active"><i class="bi bi-flag"></i> Reported Ads</a>
                    <a href="users.php"><i class="bi bi-people"></i> Users</a>
                    <a href="settings.php"><i class="bi bi-gear"></i> Settings</a>
                    <a href="pages.php"><i class="bi bi-file-text"></i> Pages</a>
                    <a href="logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a>
                </nav>
            </div>

            <!-- Main Content -->
            <div class="col-md-10 p-4">
                <h2>Reported Ads</h2>
                <hr>

                <?php if (isset($_GET['updated'])): ?>
                    <div class="alert alert-success">Report updated successfully.</div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Reports (<?php echo count($reports); ?>)</h5>
                        <form method="GET" class="d-flex gap-2">
                            <select name="status" class="form-select form-select-sm">
                                <option value="">All Status</option>
                                <option value="pending" <?php echo $statusFilter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="reviewed" <?php echo $statusFilter === 'reviewed' ? 'selected' : ''; ?>>Reviewed</option>
                                <option value="dismissed" <?php echo $statusFilter === 'dismissed' ? 'selected' : ''; ?>>Dismissed</option>
                            </select>
                            <button type="submit" class="btn btn-sm btn-primary">Filter</button>
                            <a href="reports.php" class="btn btn-sm btn-secondary">Reset</a>
                        </form>
                    </div>
                    <div class="card-body">
                        <?php if (empty($reports)): ?>
                            <p class="text-muted">No reports found.</p>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover align-middle">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Product</th>
                                            <th>Reporter</th>
                                            <th>Reason</th>
                                            <th>Status</th>
                                            <th>Created</th>
                                            <th>Admin Note / Action</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($reports as $r): ?>
                                            <tr>
                                                <td><?php echo $r['id']; ?></td>
                                                <td>
                                                    <?php if (!empty($r['product_title'])): ?>
                                                        <strong><?php echo htmlspecialchars($r['product_title']); ?></strong><br>
                                                        <span class="badge bg-secondary">
                                                            <?php echo htmlspecialchars(ucfirst($r['product_status'] ?? 'unknown')); ?>
                                                        </span>
                                                        <br>
                                                        <a href="pending.php?id=<?php echo $r['product_id']; ?>" class="btn btn-sm btn-outline-primary mt-1">
                                                            View Product
                                                        </a>
                                                    <?php else: ?>
                                                        <span class="text-muted">Product deleted</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if (!empty($r['reporter_name'])): ?>
                                                        <?php echo htmlspecialchars($r['reporter_name']); ?><br>
                                                        <small class="text-muted"><?php echo htmlspecialchars($r['reporter_phone']); ?></small>
                                                    <?php else: ?>
                                                        <span class="text-muted">Guest / Unknown</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td style="max-width: 250px;">
                                                    <?php echo nl2br(htmlspecialchars($r['reason'])); ?>
                                                </td>
                                                <td>
                                                    <?php
                                                    $badgeClass = 'secondary';
                                                    if ($r['status'] === 'pending') $badgeClass = 'warning text-dark';
                                                    elseif ($r['status'] === 'reviewed') $badgeClass = 'success';
                                                    elseif ($r['status'] === 'dismissed') $badgeClass = 'secondary';
                                                    ?>
                                                    <span class="badge bg-<?php echo $badgeClass; ?>">
                                                        <?php echo htmlspecialchars(ucfirst($r['status'])); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <?php echo date('d M Y, h:i A', strtotime($r['created_at'])); ?>
                                                </td>
                                                <td>
                                                    <form method="POST">
                                                        <input type="hidden" name="report_id" value="<?php echo $r['id']; ?>">
                                                        <div class="mb-2">
                                                            <textarea
                                                                name="admin_note"
                                                                class="form-control form-control-sm"
                                                                rows="2"
                                                                placeholder="Admin note..."
                                                            ><?php echo htmlspecialchars($r['admin_note'] ?? ''); ?></textarea>
                                                        </div>
                                                        <div class="d-flex gap-1 flex-wrap">
                                                            <button type="submit" name="status" value="pending" class="btn btn-sm btn-outline-warning">
                                                                Mark Pending
                                                            </button>
                                                            <button type="submit" name="status" value="reviewed" class="btn btn-sm btn-outline-success">
                                                                Mark Reviewed
                                                            </button>
                                                            <button type="submit" name="status" value="dismissed" class="btn btn-sm btn-outline-secondary">
                                                                Dismiss
                                                            </button>
                                                        </div>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>


