<?php
// Database Setup Script
// This script will create the database and tables if they don't exist

error_reporting(E_ALL);
ini_set('display_errors', 1);

$success = [];
$errors = [];

// Database Configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'mathbariashop');

echo "<!DOCTYPE html>";
echo "<html lang='bn'>";
echo "<head>";
echo "<meta charset='UTF-8'>";
echo "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
echo "<title>Database Setup - Mathbaria Shop</title>";
echo "<link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css' rel='stylesheet'>";
echo "<style>body { padding: 20px; }</style>";
echo "</head>";
echo "<body>";
echo "<div class='container'>";
echo "<h2>Database Setup</h2>";
echo "<hr>";

// Step 1: Connect to MySQL
echo "<h3>Step 1: Connecting to MySQL...</h3>";
try {
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    echo "<div class='alert alert-success'>✅ Connected to MySQL successfully!</div>";
    $success[] = "MySQL connection";
} catch(PDOException $e) {
    echo "<div class='alert alert-danger'>❌ Failed to connect to MySQL: " . htmlspecialchars($e->getMessage()) . "</div>";
    echo "</div></body></html>";
    exit;
}

// Step 2: Create database if not exists
echo "<h3>Step 2: Creating database...</h3>";
try {
    $conn->exec("CREATE DATABASE IF NOT EXISTS " . DB_NAME . " CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    echo "<div class='alert alert-success'>✅ Database 'mathbariashop' is ready!</div>";
    $success[] = "Database creation";
} catch(PDOException $e) {
    echo "<div class='alert alert-danger'>❌ Error creating database: " . htmlspecialchars($e->getMessage()) . "</div>";
    $errors[] = "Database creation";
}

// Step 3: Use database and create tables
echo "<h3>Step 3: Creating tables...</h3>";
try {
    $conn->exec("USE " . DB_NAME);
    
    // Read and execute schema.sql
    $schemaFile = __DIR__ . '/../database/schema.sql';
    if (file_exists($schemaFile)) {
        $sql = file_get_contents($schemaFile);
        
        // Remove CREATE DATABASE and USE statements as we already handled them
        $sql = preg_replace('/CREATE DATABASE.*?;/i', '', $sql);
        $sql = preg_replace('/USE.*?;/i', '', $sql);
        
        // Split by semicolon and execute each statement
        $statements = array_filter(array_map('trim', explode(';', $sql)));
        
        foreach ($statements as $statement) {
            if (!empty($statement) && !preg_match('/^--/', $statement)) {
                try {
                    $conn->exec($statement);
                } catch(PDOException $e) {
                    // Ignore "table already exists" errors
                    if (strpos($e->getMessage(), 'already exists') === false) {
                        echo "<div class='alert alert-warning'>⚠️ " . htmlspecialchars($e->getMessage()) . "</div>";
                    }
                }
            }
        }
        
        echo "<div class='alert alert-success'>✅ Tables created successfully!</div>";
        $success[] = "Table creation";
    } else {
        echo "<div class='alert alert-warning'>⚠️ Schema file not found at: " . htmlspecialchars($schemaFile) . "</div>";
        echo "<div class='alert alert-info'>Creating tables manually...</div>";
        
        // Create tables manually
        $tables = [
            "CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                phone VARCHAR(15) UNIQUE NOT NULL,
                password VARCHAR(255) NOT NULL,
                name VARCHAR(100) NOT NULL,
                union_name VARCHAR(50) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_phone (phone)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            "CREATE TABLE IF NOT EXISTS admins (
                id INT AUTO_INCREMENT PRIMARY KEY,
                username VARCHAR(50) UNIQUE NOT NULL,
                password VARCHAR(255) NOT NULL,
                name VARCHAR(100) NOT NULL,
                email VARCHAR(100),
                role ENUM('super_admin', 'admin', 'moderator') DEFAULT 'admin',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_username (username)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            "CREATE TABLE IF NOT EXISTS products (
                id INT AUTO_INCREMENT PRIMARY KEY,
                title VARCHAR(255) NOT NULL,
                description TEXT,
                price DECIMAL(10, 2) NOT NULL,
                category VARCHAR(50) NOT NULL,
                condition_type ENUM('new', 'used') NOT NULL DEFAULT 'used',
                union_name VARCHAR(50) NOT NULL,
                seller_id INT NOT NULL,
                seller_phone VARCHAR(15) NOT NULL,
                status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
                approved_by INT NULL,
                approved_at TIMESTAMP NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (seller_id) REFERENCES users(id) ON DELETE CASCADE,
                FOREIGN KEY (approved_by) REFERENCES admins(id) ON DELETE SET NULL,
                INDEX idx_seller (seller_id),
                INDEX idx_status (status),
                INDEX idx_union (union_name),
                INDEX idx_category (category),
                INDEX idx_created (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci"
        ];
        
        foreach ($tables as $tableSql) {
            try {
                $conn->exec($tableSql);
            } catch(PDOException $e) {
                echo "<div class='alert alert-warning'>⚠️ " . htmlspecialchars($e->getMessage()) . "</div>";
            }
        }
        
        echo "<div class='alert alert-success'>✅ Core tables created!</div>";
        $success[] = "Table creation";
    }
} catch(PDOException $e) {
    echo "<div class='alert alert-danger'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</div>";
    $errors[] = "Table creation";
}

// Step 4: Insert default admin if not exists
echo "<h3>Step 4: Setting up default admin user...</h3>";
try {
    $conn->exec("USE " . DB_NAME);
    
    // Check if admin exists
    $stmt = $conn->query("SELECT COUNT(*) FROM admins WHERE username = 'admin'");
    $adminExists = $stmt->fetchColumn() > 0;
    
    if (!$adminExists) {
        // Insert default admin (password: admin123)
        $passwordHash = password_hash('admin123', PASSWORD_DEFAULT);
        $stmt = $conn->prepare("INSERT INTO admins (username, password, name, email, role) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute(['admin', $passwordHash, 'Admin User', 'admin@mathbaria.com', 'super_admin']);
        echo "<div class='alert alert-success'>✅ Default admin user created!</div>";
        echo "<div class='alert alert-info'>Username: <strong>admin</strong><br>Password: <strong>admin123</strong></div>";
        $success[] = "Admin user creation";
    } else {
        echo "<div class='alert alert-info'>ℹ️ Admin user already exists.</div>";
        
        // Verify password
        $stmt = $conn->prepare("SELECT password FROM admins WHERE username = 'admin'");
        $stmt->execute();
        $admin = $stmt->fetch();
        
        if ($admin && password_verify('admin123', $admin['password'])) {
            echo "<div class='alert alert-success'>✅ Admin password is correct!</div>";
        } else {
            echo "<div class='alert alert-warning'>⚠️ Admin password may need to be reset.</div>";
            // Reset password
            $passwordHash = password_hash('admin123', PASSWORD_DEFAULT);
            $stmt = $conn->prepare("UPDATE admins SET password = ? WHERE username = 'admin'");
            $stmt->execute([$passwordHash]);
            echo "<div class='alert alert-success'>✅ Admin password reset to 'admin123'!</div>";
        }
    }
} catch(PDOException $e) {
    echo "<div class='alert alert-danger'>❌ Error: " . htmlspecialchars($e->getMessage()) . "</div>";
    $errors[] = "Admin user setup";
}

// Summary
echo "<hr>";
echo "<h3>Summary</h3>";
if (empty($errors)) {
    echo "<div class='alert alert-success'><strong>✅ All steps completed successfully!</strong></div>";
    echo "<p>You can now <a href='login.php' class='btn btn-primary'>Go to Login Page</a></p>";
} else {
    echo "<div class='alert alert-warning'><strong>⚠️ Some steps had warnings, but setup is mostly complete.</strong></div>";
    echo "<p>You can try <a href='login.php' class='btn btn-primary'>Logging in</a> or <a href='test_db.php' class='btn btn-secondary'>Test Database Connection</a></p>";
}

echo "</div>";
echo "</body>";
echo "</html>";
?>

