<?php
/**
 * Test upload endpoint to check file upload configuration
 * Usage: POST multipart/form-data with 'test_image' field
 */
require_once 'config.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$response = [
    'success' => false,
    'message' => '',
    'data' => []
];

try {
    // Check PHP upload settings
    $uploadMaxFilesize = ini_get('upload_max_filesize');
    $postMaxSize = ini_get('post_max_size');
    $maxFileUploads = ini_get('max_file_uploads');
    $fileUploads = ini_get('file_uploads');
    
    $response['data']['php_settings'] = [
        'upload_max_filesize' => $uploadMaxFilesize,
        'post_max_size' => $postMaxSize,
        'max_file_uploads' => $maxFileUploads,
        'file_uploads' => $fileUploads ? 'enabled' : 'disabled',
    ];
    
    // Check upload directory (UPLOAD_DIR is already absolute path)
    $uploadDirPath = UPLOAD_DIR;
    $response['data']['upload_directory'] = [
        'path' => $uploadDirPath,
        'exists' => file_exists($uploadDirPath),
        'writable' => is_writable($uploadDirPath),
        'permissions' => file_exists($uploadDirPath) ? substr(sprintf('%o', fileperms($uploadDirPath)), -4) : 'N/A',
    ];
    
    // Check if directory can be created
    if (!file_exists($uploadDirPath)) {
        if (@mkdir($uploadDirPath, 0755, true)) {
            $response['data']['upload_directory']['created'] = true;
        } else {
            $response['data']['upload_directory']['created'] = false;
            $response['data']['upload_directory']['error'] = 'Failed to create directory';
        }
    }
    
    // Check UPLOAD_DIR constant
    $response['data']['constants'] = [
        'UPLOAD_DIR' => defined('UPLOAD_DIR') ? UPLOAD_DIR : 'NOT DEFINED',
        'UPLOAD_URL' => defined('UPLOAD_URL') ? UPLOAD_URL : 'NOT DEFINED',
    ];
    
    // Check what files were received
    $response['data']['files_received'] = [
        'FILES_count' => count($_FILES),
        'FILES_keys' => array_keys($_FILES),
        'FILES_data' => $_FILES,
    ];
    
    // Check POST data
    $response['data']['post_data'] = [
        'POST_count' => count($_POST),
        'POST_keys' => array_keys($_POST),
    ];
    
    // Try to process a test upload if provided
    if (isset($_FILES['test_image']) && $_FILES['test_image']['error'] === UPLOAD_ERR_OK) {
        $testFile = $_FILES['test_image'];
        $testFilename = 'test_' . time() . '_' . basename($testFile['name']);
        $testFilepath = $uploadDirPath . '/' . $testFilename;
        
        if (move_uploaded_file($testFile['tmp_name'], $testFilepath)) {
            $response['data']['test_upload'] = [
                'success' => true,
                'filename' => $testFilename,
                'filepath' => $testFilepath,
                'size' => filesize($testFilepath),
                'url' => rtrim(UPLOAD_URL, '/') . '/' . $testFilename,
            ];
            // Delete test file
            @unlink($testFilepath);
        } else {
            $response['data']['test_upload'] = [
                'success' => false,
                'error' => 'move_uploaded_file failed',
            ];
        }
    }
    
    $response['success'] = true;
    $response['message'] = 'Upload configuration check completed';
    
} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = 'Error: ' . $e->getMessage();
    $response['data']['error'] = $e->getMessage();
    $response['data']['trace'] = $e->getTraceAsString();
}

echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

