<?php
session_start();

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../api/config.php';
$conn = getDBConnection();

// Handle approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $productId = $_POST['product_id'] ?? null;
    $action = $_POST['action'] ?? '';
    
    if ($productId && in_array($action, ['approve', 'reject'])) {
        $status = $action === 'approve' ? 'approved' : 'rejected';
        $stmt = $conn->prepare("UPDATE products SET status = ?, approved_by = ?, approved_at = NOW() WHERE id = ?");
        $stmt->execute([$status, $_SESSION['admin_id'], $productId]);
        header('Location: pending.php?success=1');
        exit;
    }
}

// Get product details if ID provided
$product = null;
if (isset($_GET['id'])) {
    $stmt = $conn->prepare("SELECT p.*, u.name as seller_name, u.phone as seller_phone 
                           FROM products p 
                           JOIN users u ON p.seller_id = u.id 
                           WHERE p.id = ?");
    $stmt->execute([$_GET['id']]);
    $product = $stmt->fetch();
    
    // Get images
    if ($product) {
        $stmt = $conn->prepare("SELECT image_path FROM product_images WHERE product_id = ? ORDER BY image_order");
        $stmt->execute([$product['id']]);
        $product['images'] = $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
}

// Get all pending products
$stmt = $conn->prepare("SELECT p.*, u.name as seller_name 
                       FROM products p 
                       JOIN users u ON p.seller_id = u.id 
                       WHERE p.status = 'pending' 
                       ORDER BY p.created_at DESC");
$stmt->execute();
$pendingProducts = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pending Products - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .sidebar {
            min-height: 100vh;
            background: #2c3e50;
        }
        .sidebar a {
            color: #ecf0f1;
            text-decoration: none;
            padding: 15px;
            display: block;
        }
        .sidebar a:hover {
            background: #34495e;
        }
        .sidebar a.active {
            background: #3498db;
        }
        .product-image {
            max-width: 200px;
            max-height: 200px;
            object-fit: cover;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-2 sidebar p-0">
                <div class="p-3">
                    <h4 class="text-white">Mathbaria Shop</h4>
                    <p class="text-white-50 small">Admin Panel</p>
                </div>
                <nav>
                    <a href="index.php"><i class="bi bi-house"></i> Dashboard</a>
                    <a href="products.php"><i class="bi bi-box"></i> Products</a>
                    <a href="pending.php" class="active"><i class="bi bi-clock-history"></i> Pending Approval</a>
                    <a href="users.php"><i class="bi bi-people"></i> Users</a>
                    <a href="logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a>
                </nav>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-10 p-4">
                <h2>Pending Products Approval</h2>
                <hr>
                
                <?php if (isset($_GET['success'])): ?>
                    <div class="alert alert-success">Product status updated successfully!</div>
                <?php endif; ?>
                
                <?php if ($product): ?>
                    <!-- Product Detail View -->
                    <div class="card mb-4">
                        <div class="card-header">
                            <h5>Product #<?php echo $product['id']; ?> - <?php echo htmlspecialchars($product['title']); ?></h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6>Images:</h6>
                                    <div class="d-flex flex-wrap gap-2 mb-3">
                                        <?php foreach ($product['images'] as $img): ?>
                                            <img src="../uploads/products/<?php echo htmlspecialchars($img); ?>" 
                                                 class="product-image rounded" alt="Product Image">
                                        <?php endforeach; ?>
                                    </div>
                                    
                                    <h6>Details:</h6>
                                    <table class="table table-bordered">
                                        <tr><th>Title:</th><td><?php echo htmlspecialchars($product['title']); ?></td></tr>
                                        <tr><th>Description:</th><td><?php echo htmlspecialchars($product['description']); ?></td></tr>
                                        <tr><th>Price:</th><td>৳<?php echo number_format($product['price'], 2); ?></td></tr>
                                        <tr><th>Category:</th><td><?php echo htmlspecialchars($product['category']); ?></td></tr>
                                        <tr><th>Condition:</th><td><?php echo ucfirst($product['condition_type']); ?></td></tr>
                                        <tr><th>Union:</th><td><?php echo htmlspecialchars($product['union_name']); ?></td></tr>
                                        <tr><th>Seller:</th><td><?php echo htmlspecialchars($product['seller_name']); ?> (<?php echo htmlspecialchars($product['seller_phone']); ?>)</td></tr>
                                        <tr><th>Posted:</th><td><?php echo date('d M Y, h:i A', strtotime($product['created_at'])); ?></td></tr>
                                    </table>
                                </div>
                            </div>
                            
                            <form method="POST" class="mt-3">
                                <input type="hidden" name="product_id" value="<?php echo $product['id']; ?>">
                                <button type="submit" name="action" value="approve" class="btn btn-success">
                                    <i class="bi bi-check-circle"></i> Approve
                                </button>
                                <button type="submit" name="action" value="reject" class="btn btn-danger">
                                    <i class="bi bi-x-circle"></i> Reject
                                </button>
                                <a href="pending.php" class="btn btn-secondary">Back to List</a>
                            </form>
                        </div>
                    </div>
                <?php endif; ?>
                
                <!-- Pending Products List -->
                <div class="card">
                    <div class="card-header">
                        <h5>All Pending Products (<?php echo count($pendingProducts); ?>)</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($pendingProducts)): ?>
                            <p class="text-muted">No pending products</p>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Title</th>
                                            <th>Price</th>
                                            <th>Category</th>
                                            <th>Seller</th>
                                            <th>Date</th>
                                            <th>Action</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($pendingProducts as $p): ?>
                                            <tr>
                                                <td><?php echo $p['id']; ?></td>
                                                <td><?php echo htmlspecialchars($p['title']); ?></td>
                                                <td>৳<?php echo number_format($p['price'], 2); ?></td>
                                                <td><?php echo htmlspecialchars($p['category']); ?></td>
                                                <td><?php echo htmlspecialchars($p['seller_name']); ?></td>
                                                <td><?php echo date('d M Y', strtotime($p['created_at'])); ?></td>
                                                <td>
                                                    <a href="pending.php?id=<?php echo $p['id']; ?>" class="btn btn-sm btn-primary">
                                                        Review
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>

